<?php  if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage libraries
* @filesource
*/ /** */

require_once 'document_retrieve.php';
require_once APPPATH.'third_party/standalone/nusoap.php';
require_once APPPATH.'third_party/standalone/nusoapmime.php';
require_once APPPATH.'third_party/standalone/simplexml_dump.php';

/**
* Parent class for the patient data SOAP calls.
*
* To extend this class:
* 	- Set up a child class ("my_child_class.php") and populate the wsdl_uri class variable.
* 	- Set up a PHP file with the XML to be sent as the request ("patient_data/_my_child_class.php")
*   - Add class variables for any of the values that need to be populated for the XML; all class variables will be passed to the XML.
*
* @package direct-project-innovation-initiative
* @subpackage libraries
*/ 
abstract class Soap_call {
	protected $CI;
	
	/**
	* URI for the WSDL for this SOAP call.  This should be set by all child classes
	* @var string
	*/
	protected $wsdl_uri;
	
	/**
	* URI for the SOAP endpoint.  If this is set in child classes, it will override the endpoint parsed from the WSDL.
	* @var string
	*/
	protected $endpoint_uri;
	
	/**
	* An array of the curl options to use when sending the SOAP call.
	* @var array
	*/	
	protected $curl_options = array();
	
	/**
	* An array of the attachments from the last call.
	* @var array
	*/	
	protected $attachments; //kind of weird having this be a class var when the response xml isn't, but we're in a hurry.  fix later.  -- MG 6/14

	/** 
	* Any messages for the user that explain why we got the response we did.  If the SOAP call returns false, this wil say something like, we couldn't connect to the server,
	* there were no documents available, etc.  This array is publically available via {@link feedback_messages()} or available as a single string via {@link feedback_message()}.
	* @var array
	*/
	protected $feedback_messages = array(); 
	
	public function __construct(){
		$this->CI = get_instance();
	}
	
	/**
	* Makes the SOAP call for this class and returns the XML response.
	*
	* @todo We'll probably need to do some sort of checking for abstract files to be able to make this work for the actual C32 data.
	* @return string
	*/
	function call(){		
		if(empty($this->wsdl_uri)) return $this->soap_connection_error('Cannot create a '.get_class($this).' SOAP call without a value for '.$this->wsdl_uri);
		
		//temporary measure, since we don't have a Connect VM with the right data on dev
		/*if(is_on_dev()){
			$this->attachments = array('9be0344c-a50f-46ef-a6c4-a14d48e3a163@example.jaxws.sun.com' => array('filename' => 'dummy_file_data'));
			ob_start();
			require '_'.mb_strtolower(get_class($this)).'_response.php';
			$buffer = ob_get_contents();
			@ob_end_clean();
			return trim($buffer);
		}*/
		
		$this->attachments = null; //clear attachments from any previous calls
		
		ini_set('allow_url_fopen', true);
		ini_set('soap.wsdl_cache_enabled',0);

		$client = new nusoap_client_mime($this->wsdl_uri, 'wsdl');
		//this will override the endpoint for the wsdl, it is required to do this if you load
		//the wsdl from a local file, because nusoap will try to use the wsdl uri as the endpoint for the send() call
		if(isset($this->endpoint_uri)) {
			$client->setEndpoint($this->endpoint_uri);
		}
		if(!empty($this->curl_options)) {
			foreach($this->curl_options as $option => $value) {
				$client->setCurlOption($option, $value);
			}
		}
		$client->soap_defencoding = 'UTF-8';
		$client->decode_utf8 = false; //when true, nusoap converts utf-8 to ISO-8859-1 using utf8_decode
		$error = $client->getError();
		if(!empty($error)) return $this->soap_connection_error('Unable to connect to SOAP server: '.$error);
		
		$result = $client->send($this->request_as_xml(), $soapaction = '', $timeout = 60, $response_timeout = 60);
		$error = $client->getError();
#		log_message('error', sprp_for_log($client->debug_str, 'NUSOAP debug string'));
		if(!empty($error)) return $this->soap_connection_error('SOAP error encountered while sending request to '.$this->wsdl_uri.': '.$error);
		
		$this->attachments = array();
		foreach($client->getAttachments() as $attachment){
			$this->attachments[strip_from_end('>', strip_from_beginning('<', $attachment['cid']))] = $attachment; //key the attachment array by cid for easy lookup
		}
		
		if(empty($client->responseData) || !$this->CI->is->xml_string($client->responseData)) 
			return $this->soap_connection_error('The SOAP request to '.$this->wsdl_uri.' did not return a valid XML response: '.$this->CI->error->describe($client->responseData));
		
		return $client->responseData;
	}	
	
	function feedback_messages(){
		return $this->feedback_messages;
	}
	
	function feedback_message(){
		return implode(" ", $this->feedback_messages);
	}
	
	/**
	* Generates the XML to be used for this request.
	* The XML should be included in a PHP file named after the class, prefixed with a '_'.
	* The class variables from this object will be passed to it to populate any variable fields.
	* @return string
	*/
	function request_as_xml(){
		ob_start();
		extract(get_object_vars($this));
		require '_'.mb_strtolower(get_class($this)).'.php';
		$buffer = ob_get_contents();
		@ob_end_clean();
		return trim($buffer);
	}	
	
	/**
	* Adds a generic server-not-available message to the user feedback messages, and adds the provided system error message to the logs for the developers.
	* @param string
	* @return boolean
	*/	
	public function soap_connection_error($system_error_message){
		if(!$this->CI->is->nonempty_string($system_error_message)) return $this->CI->error->should_be_a_nonempty_string($system_error_message);
		$this->feedback_messages[] = 'We were unable to connect to the patient data server at this time.  Please try again later, and contact an administrator if the problem persists.';
		return $this->CI->error->warning($system_error_message, $offset=1);
	}
	
	/**
	* Adds a generic no-documents-were-available message to the user feedback messages, and logs the provided message for developers if we're in debug mode.
	* To help us distinguish between server errors and genuine cases with no documents, this method will check to see if we can retrieve test documents
	* that should always be available.  If we can't, we'll trigger a {@link soap_connection_error}.
	* @param string
	* @param string
	* @return boolean
	*/
	public function no_documents_message($patient_id, $message_for_log){
		if(!$this->CI->is->nonempty_string($patient_id)) return $this->CI->error->should_be_a_nonempty_string($patient_id);
		if(!$this->CI->is->nonempty_string($message_for_log)) return $this->CI->error->should_be_a_nonempty_string($message_for_log);
		
		//if something's going wrong with the connect server, it's possible that we're getting a valid XML response that's trying to indicate an error, and we're seeing it as no documents
		//ideally, we would try to parse their errors, but we don't have enough information to know what to expect from them right now
		//so, if we don't find any documents, we'll try to grab a document that we know *should* be in the system, and if there's no data for it, we'll assume the server is down.
		if($patient_id != CONNECT_TEST_PATIENT_ID){
			$document_retrieve = new Document_retrieve(); //we need a fresh copy of the library so that we don't overlap with the existing call
			$metadata = $document_retrieve->find_c32_for_patient(CONNECT_TEST_PATIENT_ID);
			if(empty($metadata)){
				$system_error_message = 'Unable to retrieve docs for patient#'.CONNECT_TEST_PATIENT_ID.', which should always be available. The CONNECT server is probably returning errors';
				return $this->soap_connection_error($system_error_message);
			}
		}
		
		$this->feedback_messages[] = 'No documents were found for this patient.';
		log_message('debug', $message_for_log);
		return false;
	}
}